<?php
/**
 * Signing Secret Field with Generate Button
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Block\Adminhtml\System\Config;

use Magento\Backend\Block\Template\Context;
use Magento\Config\Block\System\Config\Form\Field;
use Magento\Framework\Data\Form\Element\AbstractElement;
use Magento\Framework\Encryption\EncryptorInterface;

class SigningSecretField extends Field
{
    /**
     * Template for the field
     */
    protected $_template = 'ChatData_ProductWebhook::system/config/signing_secret.phtml';

    /**
     * @var EncryptorInterface
     */
    protected $encryptor;

    /**
     * Constructor
     *
     * @param Context $context
     * @param EncryptorInterface $encryptor
     * @param array $data
     */
    public function __construct(
        Context $context,
        EncryptorInterface $encryptor,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->encryptor = $encryptor;
    }

    /**
     * Remove "Use system value" checkbox
     *
     * This disables the "Use system value" checkbox by unsetting the scope options.
     * The signing secret should be unique per installation.
     */
    public function render(AbstractElement $element): string
    {
        $element->unsScope()->unsCanUseWebsiteValue()->unsCanUseDefaultValue();
        return parent::render($element);
    }

    /**
     * Get the HTML for the element
     *
     * This replaces the default input rendering with our custom template
     */
    protected function _getElementHtml(AbstractElement $element): string
    {
        $this->setElement($element);
        return $this->_toHtml();
    }

    /**
     * Get AJAX URL for generating secret
     */
    public function getAjaxUrl(): string
    {
        return $this->getUrl('chatdata/webhook/generatesecret');
    }

    /**
     * Get the field element
     */
    public function getElement(): AbstractElement
    {
        return $this->getData('element');
    }

    /**
     * Get field HTML ID
     */
    public function getFieldId(): string
    {
        return $this->getElement()->getHtmlId();
    }

    /**
     * Get field name
     */
    public function getFieldName(): string
    {
        return $this->getElement()->getName();
    }

    /**
     * Get field value
     */
    public function getFieldValue(): string
    {
        // Get the configuration path
        $path = 'chatdata_webhook/general/signing_secret';

        // Determine scope
        $element = $this->getElement();
        $scope = $element->getScope() ?: 'default';
        $scopeId = $element->getScopeId();

        // Fetch raw value from config (this bypasses the Backend Model's masking)
        $value = (string) $this->_scopeConfig->getValue($path, $scope, $scopeId);

        // Check if value is encrypted (Magento encryption format usually contains ':')
        if (!empty($value) && strpos($value, ':') !== false) {
            try {
                return $this->encryptor->decrypt($value);
            } catch (\Exception $e) {
                // If decryption fails, return original value
                return $value;
            }
        }

        return $value;
    }
}
