<?php
/**
 * Chat Data Product Webhook Test Controller
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Controller\Adminhtml\Webhook;

use ChatData\ProductWebhook\Model\Webhook\Sender;
use ChatData\ProductWebhook\Model\Webhook\PayloadBuilder;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;

class Test extends Action
{
    /**
     * Authorization level
     */
    public const ADMIN_RESOURCE = 'ChatData_ProductWebhook::webhook_test';

    /**
     * @param Context $context
     * @param JsonFactory $resultJsonFactory
     * @param Sender $sender
     * @param ProductRepositoryInterface $productRepository
     * @param PayloadBuilder $payloadBuilder
     */
    public function __construct(
        Context $context,
        private readonly JsonFactory $resultJsonFactory,
        private readonly Sender $sender,
        private readonly ProductRepositoryInterface $productRepository,
        private readonly PayloadBuilder $payloadBuilder
    ) {
        parent::__construct($context);
    }

    /**
     * Execute test webhook
     *
     * @return \Magento\Framework\Controller\Result\Json
     */
    public function execute()
    {
        $result = $this->resultJsonFactory->create();
        $params = $this->getRequest()->getParams();
        $productId = isset($params['product_id']) ? (int)$params['product_id'] : null;
        $eventType = isset($params['event_type']) ? $params['event_type'] : 'product.created';

        try {
            if (!$productId) {
                return $result->setData([
                    'success' => false,
                    'message' => __('Product ID is required to send a test webhook.')
                ]);
            }

            // Send real product payload
            try {
                $product = $this->productRepository->getById($productId);
                $payload = $this->payloadBuilder->buildProductPayload($product, $eventType);
                $storeId = $product->getStoreId() ? (int)$product->getStoreId() : null;

                $response = $this->sender->send($payload, $eventType, $storeId, true);
            } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                return $result->setData([
                    'success' => false,
                    'message' => __('Product with ID %1 not found.', $productId)
                ]);
            }

            if ($response['success']) {
                return $result->setData([
                    'success' => true,
                    'message' => __('Test webhook sent successfully!'),
                    'details' => __('Status: %1', $response['status_code'] ?? 'OK')
                ]);
            } else {
                return $result->setData([
                    'success' => false,
                    'message' => __('Test webhook failed.'),
                    'details' => $response['message'] ?? __('Unknown error')
                ]);
            }

        } catch (\Exception $e) {
            return $result->setData([
                'success' => false,
                'message' => __('An error occurred while testing the connection.'),
                'details' => $e->getMessage()
            ]);
        }
    }
}
