<?php
/**
 * Chat Data Product Webhook Configuration Model
 *
 * Provides easy access to module configuration values
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Store\Model\ScopeInterface;

class Config
{
    /**
     * Configuration paths
     */
    private const XML_PATH_ENABLED = 'chatdata_webhook/general/enabled';
    private const XML_PATH_WEBHOOK_URL = 'chatdata_webhook/general/webhook_url';
    private const XML_PATH_SIGNING_SECRET = 'chatdata_webhook/general/signing_secret';

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param EncryptorInterface $encryptor
     */
    public function __construct(
        private readonly ScopeConfigInterface $scopeConfig,
        private readonly EncryptorInterface $encryptor
    ) {
    }

    /**
     * Check if module is enabled
     */
    public function isEnabled(?int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_ENABLED,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get webhook URL
     */
    public function getWebhookUrl(?int $storeId = null): ?string
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_WEBHOOK_URL,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get webhook signing secret (decrypted)
     */
    public function getSigningSecret(?int $storeId = null): ?string
    {
        $encrypted = $this->scopeConfig->getValue(
            self::XML_PATH_SIGNING_SECRET,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );

        return $encrypted ? $this->encryptor->decrypt($encrypted) : null;
    }

    /**
     * Check if signature validation is enabled
     * Always returns true as signature validation is mandatory
     */
    public function isSignatureEnabled(?int $storeId = null): bool
    {
        return true;
    }

    /**
     * Get HTTP timeout in seconds
     * Returns fixed value of 30
     */
    public function getTimeout(?int $storeId = null): int
    {
        return 30;
    }

}
