<?php
/**
 * Chat Data Product Webhook Product State Resolver
 *
 * Centralizes store scoping and active/visibility checks.
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Model;

use Magento\Catalog\Model\Product;

class ProductStateResolver
{
    private const ACTIVE_STATUS = 1;
    private const VISIBLE_STATUSES = [2, 3, 4];

    /**
     * @param Config $config
     */
    public function __construct(
        private readonly Config $config
    ) {
    }

    /**
     * Resolve enabled store IDs for the product.
     *
     * @param Product $product
     * @return array
     */
    public function getEnabledStoreIds(Product $product): array
    {
        $storeId = (int) $product->getStoreId();
        if ($storeId > 0) {
            return $this->config->isEnabled($storeId) ? [$storeId] : [];
        }

        $storeIds = $product->getStoreIds();
        if (is_array($storeIds) && $storeIds !== []) {
            $enabledStoreIds = [];
            foreach ($storeIds as $candidateStoreId) {
                $candidateStoreId = (int) $candidateStoreId;
                if ($candidateStoreId > 0 && $this->config->isEnabled($candidateStoreId)) {
                    $enabledStoreIds[] = $candidateStoreId;
                }
            }
            return $enabledStoreIds;
        }

        return $this->config->isEnabled(null) ? [null] : [];
    }

    /**
     * Resolve a product attribute value in the provided store scope.
     *
     * @param Product $product
     * @param string $attributeCode
     * @param int|null $storeId
     * @param bool $useOriginalFallback
     * @return int
     */
    public function getScopedAttributeValue(
        Product $product,
        string $attributeCode,
        ?int $storeId,
        bool $useOriginalFallback = false
    ): int {
        $productId = $product->getId();
        if (!$productId) {
            return (int) $product->getData($attributeCode);
        }

        $value = $product->getResource()->getAttributeRawValue(
            $productId,
            $attributeCode,
            $storeId ?? 0
        );

        if ($value === null || $value === false) {
            if ($useOriginalFallback) {
                $originalValue = $product->getOrigData($attributeCode);
                if ($originalValue !== null) {
                    return (int) $originalValue;
                }
            }

            return (int) $product->getData($attributeCode);
        }

        return (int) $value;
    }

    /**
     * Determine if a product is active and visible.
     */
    public function isActiveAndVisible(int $status, int $visibility): bool
    {
        return $status === self::ACTIVE_STATUS
            && in_array($visibility, self::VISIBLE_STATUSES, true);
    }
}
