<?php
/**
 * Chat Data Product Webhook Payload Builder
 *
 * Builds webhook payload compatible with Chat Data's magentoProductConverter
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Model\Webhook;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Eav\Api\AttributeSetRepositoryInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Psr\Log\LoggerInterface;

class PayloadBuilder
{
    private array $attributeSetNameCache = [];

    /**
     * @param ProductRepositoryInterface $productRepository
     * @param CategoryCollectionFactory $categoryCollectionFactory
     * @param AttributeSetRepositoryInterface $attributeSetRepository
     * @param StoreManagerInterface $storeManager
     * @param StockRegistryInterface $stockRegistry
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly ProductRepositoryInterface $productRepository,
        private readonly CategoryCollectionFactory $categoryCollectionFactory,
        private readonly AttributeSetRepositoryInterface $attributeSetRepository,
        private readonly StoreManagerInterface $storeManager,
        private readonly StockRegistryInterface $stockRegistry,
        private readonly Configurable $configurableType,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Build complete webhook payload for product events
     *
     * @param Product $product
     * @param string $eventType (product.created, product.updated, product.deleted)
     * @param bool $reloadProduct Whether to reload the product from repository
     * @return array
     */
    public function buildProductPayload(
        Product $product,
        string $eventType,
        bool $reloadProduct = true
    ): array
    {
        try {
            $storeId = $this->getPreferredStoreId($product->getStoreId());

            // For delete events, send minimal payload
            if ($eventType === 'product.deleted') {
                return [
                    'event_type' => $eventType,
                    'timestamp' => date('c'),
                    'store_code' => $this->storeManager->getStore($storeId)->getCode(),
                    'product' => [
                        'id' => $product->getId(),
                        'sku' => $product->getSku()
                    ]
                ];
            }

            // Load full product data (optional reload to preserve in-memory adjustments)
            $fullProduct = $reloadProduct
                ? $this->productRepository->getById(
                    $product->getId(),
                    false,
                    $storeId,
                    true
                )
                : $product;

            // Build product data with human-readable values (NO MORE MAPS!)
            $storeInfo = $this->getStoreInfo((int) $storeId);
            $productData = $this->buildProductData(
                $fullProduct,
                (string) ($storeInfo['currency_symbol'] ?? '')
            );

            return [
                'event_type' => $eventType,
                'timestamp' => date('c'),
                'store_code' => $this->storeManager->getStore($storeId)->getCode(),
                'product' => $productData,
                'store_info' => $storeInfo
            ];

        } catch (\Exception $e) {
            $this->logger->error('PayloadBuilder error: ' . $e->getMessage(), [
                'product_id' => $product->getId(),
                'event_type' => $eventType,
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Prefer a storefront store view when running in admin context.
     *
     * @param int|null $storeId
     * @return int
     */
    private function getPreferredStoreId(?int $storeId): int
    {
        if ($storeId && $storeId !== 0) {
            return (int) $storeId;
        }

        try {
            $stores = $this->storeManager->getStores();
            foreach ($stores as $store) {
                if ((int) $store->getId() !== 0 && $store->getIsActive()) {
                    return (int) $store->getId();
                }
            }
        } catch (\Exception $e) {
            $this->logger->warning('Unable to resolve storefront store view: ' . $e->getMessage());
        }

        return (int) $this->storeManager->getStore()->getId();
    }

    /**
     * Build product data with human-readable values
     *
     * @param ProductInterface $product
     * @return array
     */
    private function buildProductData(
        ProductInterface $product,
        string $currencySymbol = ''
    ): array
    {
        $formatPrice = function ($value) use ($currencySymbol) {
            if ($currencySymbol === '' || $value === null) {
                return $value;
            }
            return $currencySymbol . $value;
        };

        // Basic product information
        $data = [
            'id' => $product->getId(),
            'sku' => $product->getSku(),
            'name' => $product->getName(),
            'type_id' => $product->getTypeId(),
            'price' => $formatPrice((float) $product->getPrice()),
            'status' => (int) $product->getStatus(),
            'visibility' => (int) $product->getVisibility(),
            'weight' => $product->getWeight(),
            'created_at' => $product->getCreatedAt(),
            'updated_at' => $product->getUpdatedAt(),
        ];

        // For configurable products, use minimum child price to match server-side converter
        if ($product->getTypeId() === 'configurable') {
            $minChildPrice = $this->getConfigurableMinPrice($product);
            if ($minChildPrice !== null && $minChildPrice > 0) {
                $data['price'] = $formatPrice($minChildPrice);
            }
        }

        // Convert categories from IDs to names
        $categoryIds = $product->getCategoryIds();
        if ($categoryIds) {
            $data['categories'] = $this->convertCategoriesToNames($categoryIds);
        }

        // Convert attribute_set_id to product type name
        $attributeSetName = $this->getAttributeSetName((int) $product->getAttributeSetId());
        if ($attributeSetName) {
            $data['product_type'] = $attributeSetName;
        }

        // Convert select/multiselect attributes from IDs to human-readable labels
        $activities = $this->convertAttributeToLabels($product, 'activity');
        if ($activities) {
            $data['activities'] = $activities;
        }

        $materials = $this->convertAttributeToLabels($product, 'material');
        if ($materials) {
            $data['materials'] = $materials;
        }

        $styles = $this->convertAttributeToLabels($product, 'style_bags');
        if ($styles) {
            $data['styles'] = $styles;
        }

        $straps = $this->convertAttributeToLabels($product, 'strap_bags');
        if ($straps) {
            $data['straps'] = $straps;
        }

        $features = $this->convertAttributeToLabels($product, 'features_bags');
        if ($features) {
            $data['features'] = $features;
        }

        $colors = $this->convertAttributeToLabels($product, 'color');
        if ($colors) {
            $data['colors'] = $colors;
        }

        // HTML descriptions (send as-is, server will convert to text)
        $description = $this->getCustomAttr($product, 'description');
        if ($description) {
            $data['description_html'] = $description;
        }

        $shortDescription = $this->getCustomAttr($product, 'short_description');
        if ($shortDescription) {
            $data['short_description_html'] = $shortDescription;
        }

        // Other useful text attributes
        $brand = $this->getCustomAttr($product, 'brand');
        if ($brand) {
            $data['brand'] = $brand;
        }

        $urlKey = $this->getCustomAttr($product, 'url_key');
        if ($urlKey) {
            $data['url_key'] = $urlKey;
        }

        // Special pricing
        $specialPrice = $this->getCustomAttr($product, 'special_price');
        if ($specialPrice) {
            $data['special_price'] = (float) $specialPrice;
        }

        $specialFromDate = $this->getCustomAttr($product, 'special_from_date');
        if ($specialFromDate) {
            $data['special_from_date'] = $specialFromDate;
        }

        $specialToDate = $this->getCustomAttr($product, 'special_to_date');
        if ($specialToDate) {
            $data['special_to_date'] = $specialToDate;
        }

        // Boolean flags (convert to actual booleans)
        $isNew = $this->getCustomAttr($product, 'new');
        if ($isNew === '1') {
            $data['is_new'] = true;
        }

        $isOnSale = $this->getCustomAttr($product, 'sale');
        if ($isOnSale === '1') {
            $data['is_on_sale'] = true;
        }

        // Stock information
        try {
            $stockItem = $this->stockRegistry->getStockItemBySku($product->getSku());
            $data['stock'] = [
                'qty' => (float) $stockItem->getQty(),
                'is_in_stock' => (bool) $stockItem->getIsInStock(),
                'backorders' => (int) $stockItem->getBackorders(),
                'min_qty' => (float) $stockItem->getMinQty()
            ];
        } catch (\Exception $e) {
            $this->logger->warning('Could not load stock item for SKU: ' . $product->getSku());
        }

        // Tier prices
        $tierPrices = $product->getTierPrices();
        if ($tierPrices && count($tierPrices) > 0) {
            $data['tier_prices'] = [];
            foreach ($tierPrices as $tierPrice) {
                $data['tier_prices'][] = [
                    'qty' => (float) $tierPrice->getQty(),
                    'value' => (float) $tierPrice->getValue(),
                    'customer_group_id' => (int) $tierPrice->getCustomerGroupId()
                ];
            }
        }

        // Images (simplified structure)
        $images = $this->buildImagesList($product);
        if (!empty($images)) {
            $data['images'] = $images;
        }

        return $data;
    }

    /**
     * Get minimum price from configurable child products.
     *
     * @param ProductInterface $product
     * @return float|null
     */
    private function getConfigurableMinPrice(ProductInterface $product): ?float
    {
        try {
            $usedProducts = $this->configurableType->getUsedProducts($product);
            if (!$usedProducts || count($usedProducts) === 0) {
                return null;
            }

            $prices = [];
            foreach ($usedProducts as $child) {
                $price = (float) $child->getPrice();
                if ($price > 0) {
                    $prices[] = $price;
                }
            }

            return count($prices) > 0 ? min($prices) : null;
        } catch (\Exception $e) {
            $this->logger->warning('Could not load configurable child prices: ' . $e->getMessage(), [
                'product_id' => $product->getId(),
                'sku' => $product->getSku()
            ]);
            return null;
        }
    }

    /**
     * Get a single attribute set name by ID.
     *
     * @param int $attributeSetId
     * @return string|null
     */
    private function getAttributeSetName(int $attributeSetId): ?string
    {
        if ($attributeSetId <= 0) {
            return null;
        }

        if (array_key_exists($attributeSetId, $this->attributeSetNameCache)) {
            return $this->attributeSetNameCache[$attributeSetId];
        }

        $name = null;

        try {
            $attributeSet = $this->attributeSetRepository->get($attributeSetId);
            if ((int) $attributeSet->getEntityTypeId() === 4) {
                $name = $attributeSet->getAttributeSetName();
            }
        } catch (NoSuchEntityException $e) {
            $this->logger->warning('Attribute set not found: ' . $attributeSetId);
        } catch (\Exception $e) {
            $this->logger->error('Error loading attribute set: ' . $e->getMessage(), [
                'attribute_set_id' => $attributeSetId
            ]);
        }

        $this->attributeSetNameCache[$attributeSetId] = $name;

        return $name;
    }

    /**
     * Get store information
     *
     * @param int $storeId
     * @return array
     */
    public function getStoreInfo(int $storeId): array
    {
        try {
            $store = $this->storeManager->getStore($storeId);
            $baseUrl = $store->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_WEB);
            $currencySymbol = (string) $store->getCurrentCurrency()->getCurrencySymbol();

            // Construct Magento REST API domain
            $magentoDomain = rtrim($baseUrl, '/') . '/rest';

            return [
                'base_url' => $baseUrl,
                'currency_code' => $store->getCurrentCurrencyCode(),
                'currency_symbol' => $currencySymbol,
                'magento_domain' => $magentoDomain
            ];
        } catch (NoSuchEntityException $e) {
            $this->logger->error('Store not found: ' . $e->getMessage());
            return [
                'base_url' => '',
                'currency_code' => 'USD',
                'currency_symbol' => '',
                'magento_domain' => ''
            ];
        }
    }

    /**
     * Convert category IDs to human-readable names
     *
     * @param array $categoryIds
     * @return array
     */
    private function convertCategoriesToNames(array $categoryIds): array
    {
        if (empty($categoryIds)) {
            return [];
        }

        $categoryNames = [];

        try {
            $collection = $this->categoryCollectionFactory->create();
            $collection->addAttributeToSelect('name')
                ->addAttributeToFilter('entity_id', ['in' => $categoryIds]);

            foreach ($collection as $category) {
                $categoryNames[] = $category->getName();
            }
        } catch (\Exception $e) {
            $this->logger->error('Error converting category IDs to names: ' . $e->getMessage());
        }

        return $categoryNames;
    }

    /**
     * Convert attribute value (comma-separated IDs) to human-readable labels
     *
     * @param ProductInterface $product
     * @param string $attributeCode
     * @return array|null
     */
    private function convertAttributeToLabels(ProductInterface $product, string $attributeCode): ?array
    {
        $value = $this->getCustomAttr($product, $attributeCode);
        if (!$value) {
            return null;
        }

        try {
            // Get attribute
            $attribute = $product->getResource()->getAttribute($attributeCode);
            if (!$attribute || !$attribute->usesSource()) {
                return null;
            }

            // Convert IDs to labels
            $ids = is_array($value) ? $value : explode(',', (string) $value);
            $labels = [];

            foreach ($ids as $id) {
                $id = trim($id);
                if ($id === '') {
                    continue;
                }

                $label = $attribute->getSource()->getOptionText($id);
                if ($label && $label !== '') {
                    $labels[] = (string) $label;
                }
            }

            return !empty($labels) ? $labels : null;
        } catch (\Exception $e) {
            $this->logger->warning('Error converting attribute to labels: ' . $attributeCode . ' - ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get custom attribute value helper
     *
     * @param ProductInterface $product
     * @param string $code
     * @return mixed|null
     */
    private function getCustomAttr(ProductInterface $product, string $code)
    {
        $attr = $product->getCustomAttribute($code);
        return $attr ? $attr->getValue() : null;
    }

    /**
     * Build simplified images list
     *
     * @param ProductInterface $product
     * @return array
     */
    private function buildImagesList(ProductInterface $product): array
    {
        $images = [];
        $mediaGallery = $product->getMediaGalleryEntries();

        if ($mediaGallery) {
            foreach ($mediaGallery as $image) {
                if (!$image->isDisabled()) {
                    $images[] = [
                        'file' => $image->getFile(),
                        'label' => $image->getLabel(),
                        'position' => (int) $image->getPosition(),
                        'types' => $image->getTypes()
                    ];
                }
            }
        }

        return $images;
    }
}
