<?php
/**
 * Chat Data Product Webhook Sender
 *
 * Sends webhook HTTP requests to Chat Data endpoint
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Model\Webhook;

use ChatData\ProductWebhook\Model\Config;
use Magento\Framework\HTTP\Client\CurlFactory;
use Magento\Framework\Serialize\Serializer\Json;
use Psr\Log\LoggerInterface;

class Sender
{
    private const MAX_SEND_ATTEMPTS = 3;
    private const RETRY_DELAY_MS = 5000;
    /**
     * @param Config $config
     * @param CurlFactory $curlFactory
     * @param SignatureGenerator $signatureGenerator
     * @param Json $json
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly Config $config,
        private readonly CurlFactory $curlFactory,
        private readonly SignatureGenerator $signatureGenerator,
        private readonly Json $json,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Send webhook to Chat Data endpoint
     *
     * @param array $payload Webhook payload
     * @param string $eventType Event type (product.created, product.updated, product.deleted)
     * @param int|null $storeId Store ID
     * @param bool $isTestMode Whether this is a test webhook
     * @return array Result with 'success' (bool) and 'message' (string)
     */
    public function send(
        array $payload,
        string $eventType,
        ?int $storeId = null,
        bool $isTestMode = false
    ): array {
        // Get webhook URL
        $webhookUrl = $this->config->getWebhookUrl($storeId);
        if (!$webhookUrl) {
            if ($isTestMode) {
                return [
                    'success' => false,
                    'message' => 'Webhook URL is not configured',
                    'status_code' => null
                ];
            }
            // Intentionally skip non-test webhooks until config is complete.
            return [
                'success' => true,
                'message' => 'Webhook skipped - Webhook URL not configured',
                'status_code' => null,
                'skipped' => true
            ];
        }

        // Serialize payload to JSON
        $jsonPayload = $this->json->serialize($payload);

        // Generate signature if enabled
        $signature = null;
        if ($this->config->isSignatureEnabled($storeId)) {
            $secret = $this->config->getSigningSecret($storeId);
            if (!$secret) {
                if ($isTestMode) {
                    return [
                        'success' => false,
                        'message' => 'Signing secret is not configured',
                        'status_code' => null
                    ];
                }
                // Intentionally skip non-test webhooks until config is complete.
                return [
                    'success' => true,
                    'message' => 'Webhook skipped - Signing secret not configured',
                    'status_code' => null,
                    'skipped' => true
                ];
            }
            $signature = $this->signatureGenerator->generate($jsonPayload, $secret);
        }

        // Prepare headers
        $headers = [
            'Content-Type' => 'application/json',
            'X-Magento-Webhook-Topic' => $eventType,
            'X-Magento-Test-Mode' => $isTestMode ? 'true' : 'false',
            'X-Magento-Store-Code' => $payload['store_code'] ?? 'default'
        ];

        if ($signature) {
            $headers['X-Magento-Webhook-Signature'] = $signature;
        }

        $lastStatusCode = null;
        $lastResponseBody = null;
        $lastException = null;

        for ($attempt = 1; $attempt <= self::MAX_SEND_ATTEMPTS; $attempt++) {
            try {
                // Create HTTP client and configure
                $client = $this->curlFactory->create();
                $client->setHeaders($headers);
                $client->setTimeout($this->config->getTimeout($storeId));

                // Set CURL options for better compatibility
                $client->setOption(CURLOPT_FOLLOWLOCATION, true);
                $client->setOption(CURLOPT_RETURNTRANSFER, true);
                $client->setOption(CURLOPT_CONNECTTIMEOUT, $this->config->getTimeout($storeId));
                $client->setOption(CURLOPT_TIMEOUT, $this->config->getTimeout($storeId));

                // Disable SSL verification for HTTP (non-HTTPS) URLs
                if (strpos($webhookUrl, 'http://') === 0) {
                    $client->setOption(CURLOPT_SSL_VERIFYPEER, false);
                    $client->setOption(CURLOPT_SSL_VERIFYHOST, false);
                }

                // Send POST request
                $client->post($webhookUrl, $jsonPayload);

                // Get response
                $lastStatusCode = $client->getStatus();
                $lastResponseBody = $client->getBody();
                $lastException = null;

                // Check if successful (2xx status code)
                if ($lastStatusCode >= 200 && $lastStatusCode < 300) {
                    return [
                        'success' => true,
                        'message' => 'Webhook sent successfully',
                        'status_code' => $lastStatusCode,
                        'response' => $lastResponseBody
                    ];
                }

                if (!$this->shouldRetryStatus($lastStatusCode) || $attempt === self::MAX_SEND_ATTEMPTS) {
                    break;
                }
            } catch (\Exception $e) {
                $lastException = $e;

                if ($attempt === self::MAX_SEND_ATTEMPTS) {
                    break;
                }
            }

            usleep(self::RETRY_DELAY_MS * 1000);
        }

        if ($lastException) {
            $errorMessage = 'Exception while sending webhook: ' . $lastException->getMessage();

            $this->logger->error($errorMessage, [
                'event_type' => $eventType,
                'exception' => get_class($lastException),
                'trace' => $lastException->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => $errorMessage,
                'exception' => $lastException->getMessage()
            ];
        }

        $errorMessage = sprintf(
            'Webhook request failed with status %d: %s',
            $lastStatusCode,
            (string) $lastResponseBody
        );

        $this->logger->error($errorMessage, [
            'event_type' => $eventType,
            'status_code' => $lastStatusCode
        ]);

        return [
            'success' => false,
            'message' => $errorMessage,
            'status_code' => $lastStatusCode,
            'response' => $lastResponseBody
        ];
    }

    /**
     * Determine if a failed webhook response should be retried.
     */
    private function shouldRetryStatus(?int $statusCode): bool
    {
        if ($statusCode === null) {
            return true;
        }

        if ($statusCode === 0) {
            return true;
        }

        return $statusCode === 408
            || $statusCode === 429
            || ($statusCode >= 500 && $statusCode <= 599);
    }

}
