<?php
/**
 * Chat Data Product Webhook Signature Generator
 *
 * Generates HMAC-SHA256 signatures for webhook security
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Model\Webhook;

class SignatureGenerator
{
    /**
     * Generate HMAC-SHA256 signature for webhook payload
     *
     * @param string $payload JSON payload
     * @param string $secret Signing secret
     * @return string Base64-encoded signature
     */
    public function generate(string $payload, string $secret): string
    {
        $hash = hash_hmac('sha256', $payload, $secret, true);
        return base64_encode($hash);
    }

    /**
     * Verify webhook signature
     *
     * @param string $payload JSON payload
     * @param string $signature Received signature
     * @param string $secret Signing secret
     * @return bool True if signature is valid
     */
    public function verify(string $payload, string $signature, string $secret): bool
    {
        $expectedSignature = $this->generate($payload, $secret);

        // Use timing-safe comparison to prevent timing attacks
        return hash_equals($expectedSignature, $signature);
    }
}
