<?php
/**
 * Chat Data Product Webhook Observer - Category Delete Before
 *
 * Handles category deletion (sends product.updated for affected products)
 * When categories are deleted, products' category_ids change
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Observer;

use ChatData\ProductWebhook\Model\ProductStateResolver;
use ChatData\ProductWebhook\Model\Webhook\PayloadBuilder;
use ChatData\ProductWebhook\Model\Webhook\Sender;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Category;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Psr\Log\LoggerInterface;

class CategoryDeleteBefore implements ObserverInterface
{
    /**
     * @param ProductStateResolver $productStateResolver
     * @param PayloadBuilder $payloadBuilder
     * @param Sender $sender
     * @param ProductCollectionFactory $productCollectionFactory
     * @param ProductRepositoryInterface $productRepository
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly ProductStateResolver $productStateResolver,
        private readonly PayloadBuilder $payloadBuilder,
        private readonly Sender $sender,
        private readonly ProductCollectionFactory $productCollectionFactory,
        private readonly ProductRepositoryInterface $productRepository,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Execute observer before category is deleted
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {
        try {
            /** @var Category $category */
            $category = $observer->getEvent()->getCategory();

            if (!$category || !$category->getId()) {
                return;
            }

            // Get all products assigned to this category
            $productCollection = $this->productCollectionFactory->create();
            $productCollection->addCategoriesFilter(['in' => [$category->getId()]]);

            $eventType = 'product.updated';

            foreach ($productCollection as $product) {
                try {
                    $storeIds = $this->productStateResolver->getEnabledStoreIds($product);
                    if ($storeIds === []) {
                        continue;
                    }

                    foreach ($storeIds as $storeId) {
                        $storeIdForLoad = $storeId ?? 0;
                        $fullProduct = $this->productRepository->getById(
                            $product->getId(),
                            false,
                            $storeIdForLoad
                        );

                        // Keep consistent with direct import: only enabled + visible products
                        $status = (int) $fullProduct->getStatus();
                        $visibility = (int) $fullProduct->getVisibility();
                        if (!$this->productStateResolver->isActiveAndVisible($status, $visibility)) {
                            continue;
                        }

                        // Remove the soon-to-be-deleted category before building payload.
                        $categoryIds = $fullProduct->getCategoryIds();
                        if (is_array($categoryIds) && !empty($categoryIds)) {
                            $filteredCategoryIds = array_values(array_diff($categoryIds, [(int) $category->getId()]));
                            $fullProduct->setCategoryIds($filteredCategoryIds);
                        }

                        // Build webhook payload (exclude deleted category)
                        $payload = $this->payloadBuilder->buildProductPayload($fullProduct, $eventType, false);

                        // Send immediately
                        $result = $this->sender->send($payload, $eventType, $storeId);

                        if (!$result['success']) {
                            $this->logger->error('Category deletion webhook failed', [
                                'product_id' => $product->getId(),
                                'deleted_category_id' => $category->getId(),
                                'error' => $result['message']
                            ]);
                        }
                    }

                } catch (\Exception $e) {
                    $this->logger->error('Error processing product for category deletion', [
                        'product_id' => $product->getId(),
                        'deleted_category_id' => $category->getId(),
                        'error' => $e->getMessage()
                    ]);
                }
            }

        } catch (\Exception $e) {
            $this->logger->error('CategoryDeleteBefore observer error: ' . $e->getMessage(), [
                'category_id' => $category->getId() ?? 'unknown',
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
