<?php
/**
 * Chat Data Product Webhook Observer - Product Save After
 *
 * Handles product create and update events
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Observer;

use ChatData\ProductWebhook\Model\ProductStateResolver;
use ChatData\ProductWebhook\Model\Webhook\PayloadBuilder;
use ChatData\ProductWebhook\Model\Webhook\Sender;
use Magento\Catalog\Model\Product;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Psr\Log\LoggerInterface;

class ProductSaveAfter implements ObserverInterface
{
    /**
     * @param ProductStateResolver $productStateResolver
     * @param PayloadBuilder $payloadBuilder
     * @param Sender $sender
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly ProductStateResolver $productStateResolver,
        private readonly PayloadBuilder $payloadBuilder,
        private readonly Sender $sender,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Execute observer when product is saved
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {
        try {
            /** @var Product $product */
            $product = $observer->getEvent()->getProduct();

            if (!$product || !$product->getId()) {
                return;
            }

            $isNew = (bool) $product->getData('chatdata_is_new');
            $storeIds = $this->productStateResolver->getEnabledStoreIds($product);
            if ($storeIds === []) {
                return;
            }

            $wasActiveByStore = $product->getData('chatdata_was_active_by_store');
            $wasActiveByStore = is_array($wasActiveByStore) ? $wasActiveByStore : [];
            $wasActiveFallback = $product->getData('chatdata_was_active');

            $originalStoreId = $product->getStoreId();

            foreach ($storeIds as $storeId) {
                $status = $this->productStateResolver->getScopedAttributeValue($product, 'status', $storeId);
                $visibility = $this->productStateResolver->getScopedAttributeValue($product, 'visibility', $storeId);
                $isActive = $this->productStateResolver->isActiveAndVisible($status, $visibility);
                $storeKey = $storeId ?? 0;
                $wasActive = $wasActiveByStore[$storeKey] ?? $wasActiveFallback;
                $wasActive = $wasActive === null ? $isActive : (bool) $wasActive;

                if ($isNew) {
                    if (!$isActive) {
                        continue;
                    }
                    $eventType = 'product.created';
                } else {
                    if (!$wasActive && $isActive) {
                        $eventType = 'product.updated';
                    } elseif ($wasActive && !$isActive) {
                        $eventType = 'product.deleted';
                    } else {
                        $eventType = 'product.updated';
                    }
                }

                if ($eventType !== 'product.deleted' && !$isActive) {
                    continue;
                }

                $product->setStoreId($storeId ?? 0);

                try {
                    // Build webhook payload
                    $payload = $this->payloadBuilder->buildProductPayload($product, $eventType);

                    // Send immediately
                    $result = $this->sender->send($payload, $eventType, $storeId);

                    if (!$result['success']) {
                        $this->logger->error('Webhook failed to send', [
                            'event_type' => $eventType,
                            'product_id' => $product->getId(),
                            'error' => $result['message']
                        ]);
                    }
                } finally {
                    $product->setStoreId($originalStoreId ?? 0);
                }
            }

        } catch (\Exception $e) {
            $this->logger->error('ProductSaveAfter observer error: ' . $e->getMessage(), [
                'product_id' => $product->getId() ?? 'unknown',
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
