<?php
/**
 * Chat Data Product Webhook Observer - Product Save Before
 *
 * Captures product state needed to classify create/update/delete transitions.
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Observer;

use ChatData\ProductWebhook\Model\ProductStateResolver;
use Magento\Catalog\Model\Product;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Registry;

class ProductSaveBefore implements ObserverInterface
{
    /**
     * @param ProductStateResolver $productStateResolver
     */
    public function __construct(
        private readonly ProductStateResolver $productStateResolver,
        private readonly Registry $registry
    ) {
    }

    /**
     * Execute observer before product is saved
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {
        /** @var Product|null $product */
        $product = $observer->getEvent()->getProduct();

        if (!$product) {
            return;
        }

        $productId = $product->getId();
        if ($productId) {
            $this->registry->register($this->getProductSaveFlagKey((int) $productId), true, true);
        }

        $isNew = $product->getId() === null;
        $product->setData('chatdata_is_new', $isNew);
        if ($isNew) {
            $sku = (string) $product->getSku();
            if ($sku !== '') {
                $this->registry->register($this->getNewProductFlagKey($sku), true, true);
            }
        }

        $storeIds = $this->productStateResolver->getEnabledStoreIds($product);
        if ($storeIds === []) {
            return;
        }

        $wasActiveByStore = [];
        foreach ($storeIds as $storeId) {
            $status = $this->productStateResolver->getScopedAttributeValue($product, 'status', $storeId, true);
            $visibility = $this->productStateResolver->getScopedAttributeValue($product, 'visibility', $storeId, true);
            $wasActiveByStore[$storeId ?? 0] = $this->productStateResolver->isActiveAndVisible(
                $status,
                $visibility
            );
        }

        $product->setData('chatdata_was_active_by_store', $wasActiveByStore);
        if (count($wasActiveByStore) === 1) {
            $product->setData('chatdata_was_active', (bool) reset($wasActiveByStore));
        }
    }

    /**
     * Build registry key for product save flag.
     */
    private function getProductSaveFlagKey(int $productId): string
    {
        return 'chatdata_product_save_' . $productId;
    }

    /**
     * Build registry key for new product creation flag.
     */
    private function getNewProductFlagKey(string $sku): string
    {
        return 'chatdata_new_product_' . $sku;
    }

}
