<?php
/**
 * Chat Data Product Webhook Observer - Stock Item Save After
 *
 * Handles inventory changes (sends product.updated events)
 *
 * @category  ChatData
 * @package   ChatData_ProductWebhook
 * @author    Chat Data LLC
 * @copyright Copyright (c) 2025 Chat Data LLC
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

declare(strict_types=1);

namespace ChatData\ProductWebhook\Observer;

use ChatData\ProductWebhook\Model\ProductStateResolver;
use ChatData\ProductWebhook\Model\Webhook\PayloadBuilder;
use ChatData\ProductWebhook\Model\Webhook\Sender;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\CatalogInventory\Api\Data\StockItemInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Psr\Log\LoggerInterface;

class StockItemSaveAfter implements ObserverInterface
{
    /**
     * @param ProductStateResolver $productStateResolver
     * @param PayloadBuilder $payloadBuilder
     * @param Sender $sender
     * @param ProductRepositoryInterface $productRepository
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly ProductStateResolver $productStateResolver,
        private readonly PayloadBuilder $payloadBuilder,
        private readonly Sender $sender,
        private readonly ProductRepositoryInterface $productRepository,
        private readonly Registry $registry,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Execute observer when stock item is saved
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {
        try {
            /** @var StockItemInterface $stockItem */
            $stockItem = $observer->getEvent()->getItem();

            if (!$stockItem || !$stockItem->getProductId()) {
                return;
            }

            $productId = $stockItem->getProductId();

            if ($this->registry->registry($this->getProductSaveFlagKey((int) $productId))) {
                return;
            }

            // Load the product to get full data
            try {
                $product = $this->productRepository->getById($productId);
            } catch (NoSuchEntityException $e) {
                $this->logger->warning('Product not found for stock update', [
                    'product_id' => $productId
                ]);
                return;
            }

            $sku = (string) $product->getSku();
            if ($sku !== '' && $this->registry->registry($this->getNewProductFlagKey($sku))) {
                return;
            }

            $storeIds = $this->productStateResolver->getEnabledStoreIds($product);
            if ($storeIds === []) {
                return;
            }

            $eventType = 'product.updated';

            foreach ($storeIds as $storeId) {
                $scopedProduct = $product;
                if ($storeId !== null) {
                    try {
                        $scopedProduct = $this->productRepository->getById($productId, false, $storeId);
                    } catch (NoSuchEntityException $e) {
                        $this->logger->warning('Product not found for stock update', [
                            'product_id' => $productId
                        ]);
                        continue;
                    }
                } else {
                    $scopedProduct->setStoreId(0);
                }

                // Keep consistent with direct import: only enabled + visible products
                $status = (int) $scopedProduct->getStatus();
                $visibility = (int) $scopedProduct->getVisibility();
                if (!$this->productStateResolver->isActiveAndVisible($status, $visibility)) {
                    continue;
                }

                // Build webhook payload with updated stock data
                $payload = $this->payloadBuilder->buildProductPayload($scopedProduct, $eventType);

                // Send immediately
                $result = $this->sender->send($payload, $eventType, $storeId);

                if (!$result['success']) {
                    $this->logger->error('Stock update webhook failed to send', [
                        'event_type' => $eventType,
                        'product_id' => $productId,
                        'error' => $result['message']
                    ]);
                }
            }

        } catch (\Exception $e) {
            $this->logger->error('StockItemSaveAfter observer error: ' . $e->getMessage(), [
                'product_id' => $stockItem->getProductId() ?? 'unknown',
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Build registry key for product save flag.
     */
    private function getProductSaveFlagKey(int $productId): string
    {
        return 'chatdata_product_save_' . $productId;
    }

    /**
     * Build registry key for new product creation flag.
     */
    private function getNewProductFlagKey(string $sku): string
    {
        return 'chatdata_new_product_' . $sku;
    }
}
