<?php declare(strict_types=1);

namespace ChatData\Webhook\Service;

use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SystemConfig\SystemConfigService;

class ProductDataService
{
    private EntityRepository $productRepository;
    private EntityRepository $currencyRepository;
    private EntityRepository $salesChannelRepository;
    private SystemConfigService $systemConfigService;

    // Cached values
    private ?string $currencySymbol = null;
    private ?string $storeDomain = null;
    private bool $storeDomainLoaded = false;

    public function __construct(
        EntityRepository $productRepository,
        EntityRepository $currencyRepository,
        EntityRepository $salesChannelRepository,
        SystemConfigService $systemConfigService
    ) {
        $this->productRepository = $productRepository;
        $this->currencyRepository = $currencyRepository;
        $this->salesChannelRepository = $salesChannelRepository;
        $this->systemConfigService = $systemConfigService;
    }

    /**
     * Get the default currency symbol
     */
    public function getCurrencySymbol(Context $context): ?string
    {
        if ($this->currencySymbol !== null) {
            return $this->currencySymbol;
        }

        // Get currency from context's currency ID
        $currencyId = $context->getCurrencyId();
        $criteria = new Criteria([$currencyId]);

        $currency = $this->currencyRepository->search($criteria, $context)->first();

        $this->currencySymbol = $currency?->getSymbol();
        return $this->currencySymbol;
    }

    /**
     * Get the store domain - check plugin config first, then auto-detect from storefront.
     */
    public function getStoreDomain(Context $context): ?string
    {
        // Use flag to allow caching null values (avoid repeated queries)
        if ($this->storeDomainLoaded) {
            return $this->storeDomain;
        }

        $this->storeDomainLoaded = true;

        // Option 1: Check plugin configuration first (allows override with port, etc.)
        $configuredDomain = $this->systemConfigService->get('ChatDataWebhook.config.storeDomain');
        if ($configuredDomain && is_string($configuredDomain) && trim($configuredDomain) !== '') {
            $url = trim($configuredDomain);
            // Ensure URL has protocol
            if (!preg_match('/^https?:\/\//', $url)) {
                $url = 'https://' . $url;
            }
            // Remove trailing slash for consistent concatenation
            $this->storeDomain = rtrim($url, '/');
            return $this->storeDomain;
        }

        // Option 2: Auto-detect from storefront sales channel
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('typeId', Defaults::SALES_CHANNEL_TYPE_STOREFRONT));
        $criteria->addAssociation('domains');
        $criteria->setLimit(1);

        $salesChannel = $this->salesChannelRepository->search($criteria, $context)->first();

        if ($salesChannel === null) {
            return null;
        }

        $domains = $salesChannel->getDomains();
        if ($domains === null || $domains->count() === 0) {
            return null;
        }

        $url = $domains->first()?->getUrl();

        if ($url) {
            // Ensure URL has protocol
            if (!preg_match('/^https?:\/\//', $url)) {
                $url = 'https://' . $url;
            }
            // Remove trailing slash for consistent concatenation
            $this->storeDomain = rtrim($url, '/');
        }

        return $this->storeDomain;
    }

    /**
     * Fetch product with all associations needed by Chat Data
     */
    public function getProductWithAssociations(string $productId, Context $context): ?array
    {
        $criteria = new Criteria([$productId]);

        // Add all associations
        $criteria->addAssociations([
            'manufacturer',
            'categories',
            'tags',
            'media',
            'cover.media',
            'tax',
            'unit',
            'deliveryTime',
            'properties',
            'properties.group',
            'options',
            'options.group',
            'seoUrls',
            'configuratorSettings.option.group',
            'prices',
        ]);

        /** @var ProductEntity|null $product */
        $product = $this->productRepository->search($criteria, $context)->first();

        if ($product === null) {
            return null;
        }

        return $this->entityToArray($product);
    }

    /**
     * Enrich raw product data with fields that need API calls.
     * Server will handle formatting using shopwareProductConverter.
     */
    public function convertToFinalFormat(array $rawProduct, Context $context): array
    {
        // Add enriched fields that need API calls (server can't get these)
        $rawProduct['_currencySymbol'] = $this->getCurrencySymbol($context);
        $rawProduct['_storeDomain'] = $this->getStoreDomain($context);

        return $rawProduct;
    }

    /**
     * Convert ProductEntity to array matching Shopware API response format
     */
    private function entityToArray(ProductEntity $product): array
    {
        $data = [
            'id' => $product->getId(),
            'active' => $product->getActive(),
            'name' => $product->getName(),
            'description' => $product->getDescription(),
            'ean' => $product->getEan(),
            'manufacturerNumber' => $product->getManufacturerNumber(),
            'productNumber' => $product->getProductNumber(),
            'stock' => $product->getStock(),
            'availableStock' => $product->getAvailableStock(),
            'available' => $product->getAvailable(),
            'price' => $product->getPrice()
                ? json_decode(json_encode($product->getPrice()->jsonSerialize()), true)
                : null,
            'translated' => $product->getTranslated(),
            'createdAt' => $product->getCreatedAt()?->format(\DateTime::ATOM),
            'updatedAt' => $product->getUpdatedAt()?->format(\DateTime::ATOM),
            'releaseDate' => $product->getReleaseDate()?->format(\DateTime::ATOM),
            'minPurchase' => $product->getMinPurchase(),
            'maxPurchase' => $product->getMaxPurchase(),
            'purchaseUnit' => $product->getPurchaseUnit(),
            'referenceUnit' => $product->getReferenceUnit(),
            'shippingFree' => $product->getShippingFree(),
            'markAsTopseller' => $product->getMarkAsTopseller(),
            'weight' => $product->getWeight(),
            'width' => $product->getWidth(),
            'height' => $product->getHeight(),
            'length' => $product->getLength(),
            'restockTime' => $product->getRestockTime(),
            'keywords' => $product->getKeywords(),
            'metaTitle' => $product->getMetaTitle(),
            'metaDescription' => $product->getMetaDescription(),
            'packUnit' => $product->getPackUnit(),
            'packUnitPlural' => $product->getPackUnitPlural(),
            'customFields' => $product->getCustomFields(),
            'sales' => $product->getSales(),
            'ratingAverage' => $product->getRatingAverage(),
        ];

        // Add manufacturer
        if ($manufacturer = $product->getManufacturer()) {
            $data['manufacturer'] = [
                'id' => $manufacturer->getId(),
                'name' => $manufacturer->getName(),
                'translated' => $manufacturer->getTranslated(),
                'link' => $manufacturer->getLink(),
                'description' => $manufacturer->getDescription(),
            ];
        }

        // Add categories
        if ($categories = $product->getCategories()) {
            $data['categories'] = [];
            foreach ($categories as $category) {
                $data['categories'][] = [
                    'id' => $category->getId(),
                    'name' => $category->getName(),
                    'translated' => $category->getTranslated(),
                    'breadcrumb' => $category->getBreadcrumb(),
                    'description' => $category->getDescription(),
                    'metaTitle' => $category->getMetaTitle(),
                    'metaDescription' => $category->getMetaDescription(),
                ];
            }
        }

        // Add category tree IDs
        $data['categoryTree'] = $product->getCategoryTree() ?? [];

        // Add tags
        if ($tags = $product->getTags()) {
            $data['tags'] = [];
            foreach ($tags as $tag) {
                $data['tags'][] = [
                    'id' => $tag->getId(),
                    'name' => $tag->getName(),
                ];
            }
        }

        // Add cover media
        if ($cover = $product->getCover()) {
            $media = $cover->getMedia();
            if ($media) {
                $data['cover'] = [
                    'id' => $cover->getId(),
                    'media' => [
                        'id' => $media->getId(),
                        'url' => $media->getUrl(),
                        'alt' => $media->getAlt(),
                        'title' => $media->getTitle(),
                    ],
                ];
            }
        }

        // Add media gallery
        if ($mediaCollection = $product->getMedia()) {
            $data['media'] = [];
            foreach ($mediaCollection as $productMedia) {
                $media = $productMedia->getMedia();
                if ($media) {
                    $data['media'][] = [
                        'id' => $productMedia->getId(),
                        'position' => $productMedia->getPosition(),
                        'media' => [
                            'id' => $media->getId(),
                            'url' => $media->getUrl(),
                            'alt' => $media->getAlt(),
                            'title' => $media->getTitle(),
                        ],
                    ];
                }
            }
        }

        // Add SEO URLs
        if ($seoUrls = $product->getSeoUrls()) {
            $data['seoUrls'] = [];
            foreach ($seoUrls as $seoUrl) {
                $data['seoUrls'][] = [
                    'id' => $seoUrl->getId(),
                    'seoPathInfo' => $seoUrl->getSeoPathInfo(),
                    'isCanonical' => $seoUrl->getIsCanonical(),
                    'languageId' => $seoUrl->getLanguageId(),
                ];
            }
        }

        // Add tax
        if ($tax = $product->getTax()) {
            $data['tax'] = [
                'id' => $tax->getId(),
                'name' => $tax->getName(),
                'taxRate' => $tax->getTaxRate(),
            ];
        }

        // Add unit
        if ($unit = $product->getUnit()) {
            $data['unit'] = [
                'id' => $unit->getId(),
                'name' => $unit->getName(),
                'shortCode' => $unit->getShortCode(),
                'translated' => $unit->getTranslated(),
            ];
        }

        // Add delivery time
        if ($deliveryTime = $product->getDeliveryTime()) {
            $data['deliveryTime'] = [
                'id' => $deliveryTime->getId(),
                'name' => $deliveryTime->getName(),
                'min' => $deliveryTime->getMin(),
                'max' => $deliveryTime->getMax(),
                'unit' => $deliveryTime->getUnit(),
                'translated' => $deliveryTime->getTranslated(),
            ];
        }

        // Add properties (product characteristics)
        if ($properties = $product->getProperties()) {
            $data['properties'] = [];
            foreach ($properties as $property) {
                $propertyData = [
                    'id' => $property->getId(),
                    'name' => $property->getName(),
                    'groupId' => $property->getGroupId(),
                    'translated' => $property->getTranslated(),
                ];

                if ($group = $property->getGroup()) {
                    $propertyData['group'] = [
                        'id' => $group->getId(),
                        'name' => $group->getName(),
                        'translated' => $group->getTranslated(),
                    ];
                }

                $data['properties'][] = $propertyData;
            }
        }

        // Add options (variant options)
        if ($options = $product->getOptions()) {
            $data['options'] = [];
            foreach ($options as $option) {
                $optionData = [
                    'id' => $option->getId(),
                    'name' => $option->getName(),
                    'groupId' => $option->getGroupId(),
                    'translated' => $option->getTranslated(),
                ];

                if ($group = $option->getGroup()) {
                    $optionData['group'] = [
                        'id' => $group->getId(),
                        'name' => $group->getName(),
                        'translated' => $group->getTranslated(),
                    ];
                }

                $data['options'][] = $optionData;
            }
        }

        // Add configurator settings (for variant products)
        if ($configuratorSettings = $product->getConfiguratorSettings()) {
            $data['configuratorSettings'] = [];
            foreach ($configuratorSettings as $setting) {
                $settingData = [
                    'id' => $setting->getId(),
                    'optionId' => $setting->getOptionId(),
                    'position' => $setting->getPosition(),
                ];

                if ($option = $setting->getOption()) {
                    $settingData['option'] = [
                        'id' => $option->getId(),
                        'name' => $option->getName(),
                        'groupId' => $option->getGroupId(),
                        'translated' => $option->getTranslated(),
                    ];

                    if ($group = $option->getGroup()) {
                        $settingData['option']['group'] = [
                            'id' => $group->getId(),
                            'name' => $group->getName(),
                            'translated' => $group->getTranslated(),
                        ];
                    }
                }

                $data['configuratorSettings'][] = $settingData;
            }
        }

        // Add advanced prices (tier pricing)
        if ($prices = $product->getPrices()) {
            $data['prices'] = [];
            foreach ($prices as $price) {
                $data['prices'][] = [
                    'id' => $price->getId(),
                    'quantityStart' => $price->getQuantityStart(),
                    'quantityEnd' => $price->getQuantityEnd(),
                    'price' => $price->getPrice()
                        ? json_decode(json_encode($price->getPrice()->jsonSerialize()), true)
                        : null,
                ];
            }
        }

        return $data;
    }
}
