<?php declare(strict_types=1);

namespace ChatData\Webhook\Service;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;
use Shopware\Core\System\SystemConfig\SystemConfigService;

class WebhookService
{
    private const CONFIG_WEBHOOK_URL = 'ChatDataWebhook.config.webhookUrl';
    private const CONFIG_SIGNING_SECRET = 'ChatDataWebhook.config.signingSecret';

    private SystemConfigService $systemConfigService;
    private LoggerInterface $logger;
    private Client $httpClient;
    private ProductDataService $productDataService;

    public function __construct(
        SystemConfigService $systemConfigService,
        LoggerInterface $logger,
        ProductDataService $productDataService
    ) {
        $this->systemConfigService = $systemConfigService;
        $this->logger = $logger;
        $this->productDataService = $productDataService;
        $this->httpClient = new Client([
            'timeout' => 30,
            'connect_timeout' => 10,
        ]);
    }

    /**
     * Check if webhook is configured
     */
    public function isConfigured(): bool
    {
        $webhookUrl = $this->systemConfigService->getString(self::CONFIG_WEBHOOK_URL);
        $this->logger->info('ChatData: Checking webhook configuration', [
            'configKey' => self::CONFIG_WEBHOOK_URL,
            'webhookUrl' => $webhookUrl ?: '(empty)',
            'isConfigured' => !empty($webhookUrl),
        ]);
        return !empty($webhookUrl);
    }

    /**
     * Send webhook to Chat Data endpoint
     */
    public function send(string $topic, array $payload): void
    {
        $webhookUrl = $this->systemConfigService->getString(self::CONFIG_WEBHOOK_URL);
        $signingSecret = $this->systemConfigService->getString(self::CONFIG_SIGNING_SECRET);

        if (empty($webhookUrl)) {
            return;
        }

        $jsonPayload = json_encode($payload, JSON_THROW_ON_ERROR);
        $signature = base64_encode(
            hash_hmac('sha256', $jsonPayload, $signingSecret, true)
        );

        $headers = [
            'Content-Type' => 'application/json',
            'x-shopware-webhook-topic' => $topic,
            'x-shopware-webhook-signature' => $signature,
        ];

        $this->logger->info('Sending webhook', [
            'topic' => $topic,
            'url' => $webhookUrl,
        ]);

        try {
            $response = $this->httpClient->post($webhookUrl, [
                'headers' => $headers,
                'body' => $jsonPayload,
            ]);

            $this->logger->info('Webhook sent successfully', [
                'topic' => $topic,
                'statusCode' => $response->getStatusCode(),
            ]);
        } catch (GuzzleException $e) {
            $this->logger->error('Webhook delivery failed', [
                'topic' => $topic,
                'url' => $webhookUrl,
                'error' => $e->getMessage(),
                'code' => $e->getCode(),
            ]);
            // Re-throw so caller knows it failed
            throw $e;
        }
    }
}