<?php declare(strict_types=1);

namespace ChatData\Webhook\Subscriber;

use ChatData\Webhook\Service\ProductDataService;
use ChatData\Webhook\Service\WebhookService;
use Shopware\Core\Defaults;
use Shopware\Core\Content\Product\ProductEvents;
use Shopware\Core\Framework\DataAbstractionLayer\Event\EntityDeletedEvent;
use Shopware\Core\Framework\DataAbstractionLayer\Event\EntityWrittenEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Psr\Log\LoggerInterface;

class ProductSubscriber implements EventSubscriberInterface
{
    private WebhookService $webhookService;
    private ProductDataService $productDataService;
    private LoggerInterface $logger;

    public function __construct(
        WebhookService $webhookService,
        ProductDataService $productDataService,
        LoggerInterface $logger
    ) {
        $this->webhookService = $webhookService;
        $this->productDataService = $productDataService;
        $this->logger = $logger;
    }

    public static function getSubscribedEvents(): array
    {
        return [
            ProductEvents::PRODUCT_WRITTEN_EVENT => 'onProductWritten',
            ProductEvents::PRODUCT_DELETED_EVENT => 'onProductDeleted',
        ];
    }

    public function onProductWritten(EntityWrittenEvent $event): void
    {
        $this->logger->info('ChatData: onProductWritten triggered', [
            'entityName' => $event->getEntityName(),
            'writeResultsCount' => count($event->getWriteResults()),
        ]);

        if (!$this->webhookService->isConfigured()) {
            $this->logger->warning('ChatData: Webhook not configured, skipping');
            return;
        }

        $this->logger->info('ChatData: Webhook is configured, processing events');

        $context = $event->getContext();

        foreach ($event->getWriteResults() as $writeResult) {
            $productId = $this->extractProductId($writeResult->getPrimaryKey());

            if ($productId === null) {
                continue;
            }

            // Determine if this is a create or update
            $operation = $writeResult->getOperation();
            $isInsert = $operation === 'insert';
            $topic = $isInsert ? 'product.created' : 'product.updated';

            try {
                // Fetch full product data with all associations
                $productData = $this->productDataService->getProductWithAssociations(
                    $productId,
                    $context
                );

                if ($productData === null) {
                    $this->logger->warning('Product not found for webhook', [
                        'productId' => $productId,
                        'topic' => $topic,
                    ]);
                    continue;
                }

                // For CREATE operations, skip inactive products (they haven't been published yet)
                // For UPDATE operations, always send (captures activation, deactivation, and changes)
                if ($isInsert && !($productData['active'] ?? true)) {
                    $this->logger->info('Skipping inactive product creation', [
                        'productId' => $productId,
                    ]);
                    continue;
                }

                // Convert to final format with currency and domain
                $formattedData = $this->productDataService->convertToFinalFormat($productData, $context);

                $this->webhookService->send($topic, $formattedData);

                $this->logger->info('Webhook sent successfully', [
                    'productId' => $productId,
                    'topic' => $topic,
                ]);
            } catch (\Throwable $e) {
                $this->logger->error('Failed to send webhook', [
                    'productId' => $productId,
                    'topic' => $topic,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    public function onProductDeleted(EntityDeletedEvent $event): void
    {
        if (!$this->webhookService->isConfigured()) {
            return;
        }

        foreach ($event->getWriteResults() as $writeResult) {
            $productId = $this->extractProductId($writeResult->getPrimaryKey());

            if ($productId === null) {
                continue;
            }

            try {
                // For deletes, we only need to send the product ID
                $payload = [
                    'id' => $productId,
                ];

                $this->webhookService->send('product.deleted', $payload);

                $this->logger->info('Delete webhook sent successfully', [
                    'productId' => $productId,
                ]);
            } catch (\Throwable $e) {
                $this->logger->error('Failed to send delete webhook', [
                    'productId' => $productId,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    private function extractProductId($primaryKey): ?string
    {
        if (is_string($primaryKey)) {
            return $primaryKey;
        }

        if (!is_array($primaryKey)) {
            return null;
        }

        $productId = $primaryKey['id'] ?? null;

        if (!is_string($productId)) {
            return null;
        }

        $versionId = $primaryKey['versionId'] ?? null;

        if ($versionId !== null && $versionId !== Defaults::LIVE_VERSION) {
            return null;
        }

        return $productId;
    }
}
